/*
 *	mkcct.c - Build Code Coversion Table (CCT)
 */

#include <stdio.h>
#include <stdlib.h>
#include <windows.h>

#include <xfont.h>

#define MAX_CCTSIZE	(64*1024)

#define MAX_CCT	64

#ifdef DEBUG
#define DPRINTF	if (Verbose) printf
#else
#define printf
#endif

typedef struct CCT {
	XCCTENT	ent;
	XCCT	cct[MAX_CCT];
} CCT, *PCCT;

static char	*cmdname = "mkcct";

static char	*Usage[] = {
	"Usage: %s [options] {index-file} {cct-file}",
	"	-i#:	CCT index (#=0-255)",
	"	-v:	verbose output",
	NULL,
};

static int	CCTindex;
static BOOL	Verbose;

static int	CurLine;
static int	CCTOffset;

/*
 *	fatal
 */
static void fatal(char *s)
{
	fprintf(stderr, "%s: ", cmdname);
	perror(s);
	fprintf(stderr, "\n");
	exit(1);
}

/*
 *	error
 */
static void error(char *fmt, void *v)
{
	fprintf(stderr, "%s: ", cmdname);
	fprintf(stderr, fmt, v);
	fprintf(stderr, " in line %d", CurLine);
	fprintf(stderr, "\n");
	exit(1);
}

/*
 *	BuildCCTHeader
 */
static void BuildCCTHeader(PXCCTHEAD pxc)
{
	memcpy(pxc->sig, XCCTH_SIG, sizeof(pxc->sig));
	pxc->ver = XCCTH_VER;
	pxc->size = sizeof(*pxc);
	pxc->index = CCTindex;
	memset(pxc->reserved, 0, sizeof(pxc->reserved));
}

/*
 *	CommitCCT
 */
int CommitCCT(int size)
{
	int		offset;

	if (CCTOffset + size > MAX_CCTSIZE)
		error("table overflow.", NULL);
	offset = CCTOffset;
	CCTOffset += size;

	return offset;
}

/*
 *	mkcct
 */
static void mkcct(char *idxname, char *cctname)
{
	FILE		*fpidx;
	FILE		*fpcct;
	int		scode, ecode;
	int		s0, s1, s2, s3;
	int		e3;
	int		i;
	int		fontoffset;
	PCCT		**pKlo;		// 80-9F
	PCCT		**pKhi;		// E0-FF
	PCCT		**pk;
	PCCT		**ps1, *ps2;
	PCCT		pcct;
	PXCCT		pxcct;
	char		rbuf[256];
	XCCTHEAD	ccthead;

	char		*pCCTbase;
	int		ilo;
	int		ihi;
	int		i1, i2, i3;
	int		x;
	int		j, k;
	int		size;
	int		ix;
	PXCCTENT	px;
	PXCCT		pc;

	/*
	 *	Check text file and params
	 */

	if ((fpidx = fopen(idxname, "r")) == NULL)
		fatal(idxname);

	/*
	 *	Build CCT tree
	 */

	CurLine = 0;
	fontoffset = 0;

	if (!(pKlo = (PCCT **)calloc(32, sizeof(PCCT *))))
		error("no memory.", NULL);
	if (!(pKhi = (PCCT **)calloc(32, sizeof(PCCT *))))
		error("no memory.", NULL);

	while (fgets(rbuf, sizeof rbuf, fpidx)) {
		CurLine++;

		if (rbuf[0] == '#' ||	// comment
			rbuf[0] == '\n')
			continue;

		sscanf(rbuf, "%x %x", &scode, &ecode);
		DPRINTF("scode:%x ecode:%x\n", scode, ecode);
		
		if (scode > ecode)
			error("Illegal code range", NULL);

		while (scode <= ecode) {

			/*
			 *	level 1
			 */

			s0 = HIBYTE(scode) >> 5;
			s1 = HIBYTE(scode) & 0x1f;

			switch (s0) {
			default:
				error("Not a kanji code(%d)", (void*)s0);
				// NOTREACHED
			case 4:
				ps1 = &pKlo[s1];
				break;
			case 7:
				ps1 = &pKhi[s1];
				break;
			}
			if (!*ps1)
				if (!(*ps1 = (PCCT*)calloc(4, sizeof(PCCT))))
					error("no memory", NULL);

			/*
			 *	level 2
			 */

			s2 = LOBYTE(scode) >> 6;
			ps2 = &(*ps1)[s2];
			if (!*ps2)
				if (!(*ps2 = (PCCT)calloc(1, sizeof(CCT))))
					error("no memory", NULL);

			/*
			 *	level 3
			 */

			pcct = *ps2;
			if (!pcct)
				error("Internal error.", NULL);

			s3 = LOBYTE(scode) & 0x3f;
			e3 = LOBYTE(ecode) & 0x3f;
			if ((ecode - scode) > (e3 - s3))
				e3 = 0x3f;
				
			i = pcct->ent.count++;
			pxcct = &pcct->cct[i];
			pxcct->start = s3;
			pxcct->end = e3;
			pxcct->offset = fontoffset;

			DPRINTF("  %d:%d:%d[%d]=%x:%x(%x)\n", s1, s2, s3, i,
				s3, e3, fontoffset);

			fontoffset += (e3 - s3) + 1;

			/*
			 *	next
			 */

			s3 += (e3 - s3) + 1;
			if (s3 >= 64) {
				s3 = 0;
				s2++;
				if (s2 >= 4) {
					s2 = 0;
					s1++;
				}
			}

			scode = MAKEWORD(s3 | (s2 << 6), s1 | (s0 << 5));
		}
	}

#if 0
	if (Verbose) {
		for (x = 0x80; x < 0xff; x += 0x60) {
			pk = (x == 0x80) ? pKlo : pKhi;
			for (i = 0; i < 32; i++) {
				DPRINTF("%2x: ", i + x);
				ps1 = &pk[i];
				if (!*ps1) {
					DPRINTF("<null>\n");
					continue;
				}
				DPRINTF("\n");
				for (j = 0; j < 4; j++) {
					DPRINTF("  %2x: ", j);
					ps2 = &(*ps1)[j];
					if (!*ps2) {
						DPRINTF("<null>\n");
						continue;
					}
					DPRINTF("\n");
					pcct = *ps2;
					for (k = 0; k < pcct->ent.count; k++) {
						DPRINTF("    %2x: ", k);
						pxcct = &pcct->cct[k];
						DPRINTF("      %x:%x:%x\n",
							pxcct->start,
							pxcct->end,
							pxcct->offset);
					}
				}
			}
		}
	}
#endif

	/*
	 *	Build CCT image
	 */

	if (!(pCCTbase = (char *)calloc(1, MAX_CCTSIZE)))
		error("malloc failed.", NULL);

	ilo = CommitCCT(sizeof(u16) * 32);
	ihi = CommitCCT(sizeof(u16) * 32);

	for (x = 0x80; x < 0xff; x += 0x60) {
		i1 = (x == 0x80) ? ilo : ihi;
		pk = (x == 0x80) ? pKlo : pKhi;
		for (i = 0; i < 32; i++) {
			ps1 = &pk[i];
			if (!*ps1)
				continue;

			i2 = CommitCCT(sizeof(u16) * 4);
			ix = i1 + i * sizeof(u16);
			*(u16 *)(pCCTbase + ix) = i2;
			for (j = 0; j < 4; j++) {
				ps2 = &(*ps1)[j];
				if (!*ps2)
					continue;

				pcct = *ps2;
				size = sizeof(XCCTENT) +
					sizeof(XCCT) * pcct->ent.count;
				i3 = CommitCCT(size);
				ix = i2 + j * sizeof(u16);
				if (*(u16 *)(pCCTbase + ix) == 0)
					*(u16 *)(pCCTbase + ix) = i3;

				px = (PXCCTENT)&pCCTbase[i3];
				pc = (PXCCT)(pCCTbase + i3 + sizeof(*px));
				px->count = pcct->ent.count;
				for (k = 0; k < px->count; k++) {
					pc[k].start = pcct->cct[k].start;
					pc[k].end = pcct->cct[k].end;
					pc[k].offset = pcct->cct[k].offset;
				}
			}
		}
	}

	if (Verbose) {
		u16		*p1, *p2;

		for (x = 0x80; x < 0xff; x += 0x60) {
			i1 = (x == 0x80) ? 0 : sizeof(u16) * 32;
			for (i = 0; i < 32; i++) {
				DPRINTF("%2x: ", i + x);
				ix = i1 + i * sizeof(u16);
				p1 = (u16 *)(pCCTbase + ix);
				if (!*p1) {
					DPRINTF("<null>\n");
					continue;
				}
				DPRINTF("\n");
				p2 = (u16 *)(pCCTbase + *p1);
				for (j = 0; j < 4; j++) {
					DPRINTF("  %2x: ", j);
					if (!p2[j]) {
						DPRINTF("<null>\n");
						continue;
					}
					DPRINTF("\n");
					pcct = (PCCT)(pCCTbase + p2[j]);
					for (k = 0; k < pcct->ent.count; k++) {
						DPRINTF("    %2x: ", k);
						pxcct = &pcct->cct[k];
						DPRINTF("      %x:%x:%x\n",
							pxcct->start,
							pxcct->end,
							pxcct->offset);
					}
				}
			}
		}
	}

	/*
	 *	Write CCT
	 */

	if ((fpcct = fopen(cctname, "wb")) == NULL)
		fatal(cctname);
	BuildCCTHeader(&ccthead);
	if (fwrite(&ccthead, sizeof ccthead, 1, fpcct) != 1)
		fatal(cctname);
	if (fwrite(pCCTbase, CCTOffset, 1, fpcct) != 1)
		fatal(cctname);
	fclose(fpcct);

	/*
	 *	Finishing
	 */

	fclose(fpidx);
}

/*
 *	usage
 */
static void usage()
{
	char		**p;

	p = Usage;
	fprintf(stderr, *p++, cmdname);
	fprintf(stderr, "\n");
	for (; *p; p++)
		fprintf(stderr, "%s\n", *p);

	exit(1);
}

/*
 *	main
 */
void main(int argc, char* argv[])
{
	/*
	 *	Get options
	 */

	argc--, argv++;
	while (argc && **argv == '-') switch (argv[0][1]) {
	default:
		usage();
		// NOTREACHED
	case 'i':
		if (argv[0][2] == '\0')
			usage();
		CCTindex = atoi(&argv[0][2]);
		if (CCTindex < 0 || CCTindex > 255)
			usage();
		argc--, argv++;
		break;
	case 'v':
		Verbose = TRUE;
		argc--, argv++;
		break;
	}

	if (argc != 2)
		usage();

	mkcct(argv[0], argv[1]);
}

